//
//  ColorWheel.swift
//  Do It
//
//  Created by Jim Dovey on 2/4/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

// START:LoupePreference
fileprivate struct LoupeLocationPreferenceKey: PreferenceKey {
    typealias Value = Anchor<CGPoint>? // <label id="code.ch5.loupe.pref.value" />
    static var defaultValue: Anchor<CGPoint>? = nil // <label id="code.ch5.loupe.pref.default" />
    static func reduce(
        value: inout Anchor<CGPoint>?,
        nextValue: () -> Anchor<CGPoint>?
    ) {
        if value == nil { // <label id="code.ch5.loupe.pref.reduce" />
            value = nextValue()
        }
    }
}
// END:LoupePreference

// START:BrightnessBar
fileprivate struct BrightnessBar<Value: ColorInfo>: View {
    @Binding var color: Value

    // END:BrightnessBar
    // START:BarGradientDefinition
    var gradient: Gradient {
        let (h, s, _) = color.hsb
        return Gradient(colors: [
            Color(hue: h, saturation: s, brightness: 1),
            Color(hue: h, saturation: s, brightness: 0)
        ])
    }
    // END:BarGradientDefinition

    // START:BarSelection
    func selectionOffset(_ proxy: GeometryProxy) -> CGSize {
        CGSize(width: 0,
               height: CGFloat(1.0-color.brightness) * proxy.size.height - 5)
    }
    // END:BarSelection

    // START:BrightnessBar
    var body: some View {
        GeometryReader { proxy in // <label id="code.ch5.bar.geometry" />
            // START:BrightnessGesture
            ZStack(alignment: .top) {
                // <literal:elide> ... </literal:elide>
                // END:BrightnessGesture
                // END:BrightnessBar
                // START:BarGradient
                LinearGradient(gradient: self.gradient,
                               startPoint: .top,
                               endPoint: .bottom)
                    .border(Color.white)
                // END:BarGradient

                // START:BarSlider
                self.color.uiColor
                    .border(Color.white)
                    .frame(height: 10)
                    .offset(self.selectionOffset(proxy))
                // END:BarSlider
                // START:BrightnessBar
                // START:BrightnessGesture
            }
            // END:BrightnessBar
            .gesture(DragGesture(minimumDistance: 0).onChanged {
                let value = 1.0 - Double($0.location.y / proxy.size.height)
                self.color.brightness = min(max(0.0, value), 1.0)
            })
            // START:BrightnessBar
            // END:BrightnessGesture
        }
    }
}
// END:BrightnessBar

// START:ColorWheel
struct ColorWheel<Value: ColorInfo>: View {
    @Binding var color: Value
    
    @State private var dragging = false
    @State private var loupeLocation: CGPoint = .zero

    // END:ColorWheel
    // START:HueGradient
    private var wheelGradient: AngularGradient {
        let (_, _, b) = color.hsb
        let stops: [Gradient.Stop] = stride(from: 0.0, through: 1.0, by: 0.01).map {
            Gradient.Stop(color: Color(hue: $0, saturation: 1, brightness: b),
                          location: CGFloat($0))
        }
        let gradient = Gradient(stops: stops)
        return AngularGradient(gradient: gradient, center: .center,
                               angle: .degrees(360))
    }
    // END:HueGradient

    // START:SaturationGradient
    private func fadeGradient(radius: CGFloat) -> RadialGradient {
        let (_, _, b) = color.hsb
        let fadeColor = Color(hue: 0, saturation: 0, brightness: b)
        let gradient = Gradient(colors: [fadeColor, fadeColor.opacity(0)])
        return RadialGradient(gradient: gradient, center: .center,
                              startRadius: 0, endRadius: radius)
    }
    // END:SaturationGradient

    // START:ColorWheel
    // START:BrightnessBarAspect
    var body: some View {
        HStack(spacing: 16) {
            // END:ColorWheel
            // <literal:elide> ... </literal:elide>
            // END:BrightnessBarAspect
            // START:ColorWheel
            // START:WheelAspect
            GeometryReader { proxy in
                // <literal:elide> ... </literal:elide>
                // END:ColorWheel
                // END:WheelAspect
                // START:WheelView
                ZStack {
                    self.wheelGradient
                        .overlay(self.fadeGradient(radius: proxy.size.width/2))
                        .clipShape(Circle())
                        .overlay(Circle().stroke(Color.white))
                    // END:WheelView

                    // START:ShowSelection
                    if !self.dragging {
                        Rectangle()
                            .fill(self.color.uiColor)
                            .overlay(Rectangle().stroke(Color.white, lineWidth: 1))
                            .frame(width: 16, height: 16)
                            .offset(HSB.unitOffset(for: self.color, within: proxy.size))
                    }
                    // END:ShowSelection
                    // START:WheelView
                }
                // END:WheelView
                // START:WheelGesture
                .gesture(
                    DragGesture(minimumDistance: 0).onChanged {
                        self.dragging = true // <label id="code.ch5.wheel.drag1" />
                        self.loupeLocation = $0.location
                            .boundedInCircle(radius: proxy.size.width/2) // <label id="code.ch5.wheel.drag3" />
                        self.assignColor(at: self.loupeLocation, in: proxy)
                    }
                    .onEnded { _ in
                        self.dragging = false  // <label id="code.ch5.wheel.drag2" />
                        self.assignColor(at: self.loupeLocation, in: proxy)
                    }
                )
                // END:WheelGesture
                // START:AnchorPreference
                .anchorPreference(key: LoupeLocationPreferenceKey.self,
                                  value: .point(self.loupeLocation),
                                  transform: { $0 })
                // END:AnchorPreference
                // START:ColorWheel
                // START:WheelAspect
            }
            // END: ColorWheel
            .aspectRatio(contentMode: .fit)
            .modifier(DoubleShadow())
            // END:WheelAspect
            // START:ShowLoupe
            .overlayPreferenceValue(LoupeLocationPreferenceKey.self) { anchor in
                GeometryReader { geometry in // <label id="code.ch5.loupe.geometry" />
                    self.buildLoupe(geometry, anchor) // <label id="code.ch5.loupe.build" />
                        .opacity(self.dragging ? 1 : 0) // <label id="code.ch5.loupe.opacity" />
                }
            }
            // END:ShowLoupe

            // START:ShowBrightnessBar
            BrightnessBar(color: self.$color)
                .padding(.vertical, 30)
                .frame(maxWidth: 30)
                .modifier(DoubleShadow())
                .zIndex(-1) // <label id="code.ch5.bar.zIndex" />
            // END:ShowBrightnessBar
            // START:ColorWheel
            // START:BrightnessBarAspect
        }
        // END:ColorWheel
        // START_HIGHLIGHT
        .aspectRatio(1.125, contentMode: .fit)
        // END_HIGHLIGHT
        // START:ColorWheel
    }
    // END:ColorWheel
    // END:BrightnessBarAspect

    // START:BuildLoupe
    private func buildLoupe(
        _ geometry: GeometryProxy,
        _ anchor: Anchor<CGPoint>?
    ) -> some View {
        let location = anchor != nil ? geometry[anchor!] : .zero // <label id="code.ch5.loupe.location.real" />
        let unitLocation = location.centeredUnit(within: geometry.size) // <label id="code.ch5.loupe.location.unit" />

        return Circle()
            .fill(HSB.uiColor(at: unitLocation, basedOn: color)) // <label id="code.ch5.loupe.location.color" />
            .overlay(Circle().stroke(Color.white, lineWidth: 1))
            .frame(width: 70, height: 70)
            .modifier(DoubleShadow())
            .offset(x: location.x - 35, y: location.y - 35) // <label id="code.ch5.loupe.location.offset" />
    }
    // END:BuildLoupe

    // START:AssignColor
    private func assignColor(at location: CGPoint, in geometry: GeometryProxy) {
        let unitLocation = location.centeredUnit(within: geometry.size)
        HSB.updateColor(&color, at: unitLocation)
    }
    // END:AssignColor
    // START:ColorWheel
}
// END:ColorWheel

struct ColorWheel_Previews: PreviewProvider {
    static var previews: some View {
        // START:Preview
        StatefulPreviewWrapper(TodoItemList.Color.purple) { binding in
            VStack {
                ColorWheel(color: binding)
                // END:Preview

                // START:PreviewFeedback
                RoundedRectangle(cornerRadius: 12)
                    .fill(binding.wrappedValue.uiColor)
                    .overlay(RoundedRectangle(cornerRadius: 12)
                        .stroke(Color.white))
                    .frame(width: 300, height: 60)
                    .modifier(DoubleShadow())
                    .padding(.top)
                    .zIndex(-1)
                // END:PreviewFeedback
                // START:Preview
            }
        }
        // END:Preview
    }
}
